import random
import time
from dataclasses import dataclass, field
from typing import Dict, List, Optional
import argparse
from enum import Enum
import re
import json
import logging
from pathlib import Path
from io_utils import sclear, sinput, sprint, init_sumulator
from pathlib import Path

manual_text = """
"""


SLEEP_TIME = 0 #декорация, задержка отображения по-умолчанию 2

class BaseMode(Enum):
    FAST = 1
    NORMAL = 2
    SLOW = 3
    
    @classmethod
    def from_string(cls, mode_str: str):
        mapping = {
            'FAST': cls.FAST,
            'NORMAL': cls.NORMAL,
            'SLOW': cls.SLOW
        }
        return mapping.get(mode_str.upper(), cls.NORMAL)
    
BaseModeCaptions = ["БЫСТРЫЙ", "УНИВЕРСАЛ", "МЕДЛЕННЫЙ"]

class SpecMode(Enum):
    CALIBRATE = 1
    STABILIZE = 2
    PAUSE_POW = 3
    
SpecModeCaptions = ["КАЛИБРОВКА", "СТАБИЛИЗАЦИЯ", "ПАУЗА-ЗАРЯДКА"]

@dataclass
class PositiveEvent:
    """Модель позитивного события"""
    name: str
    description: str
    effects: Dict[str, int]  # Изменения параметров
    min_score: int  # Минимальный счет для активации
    probability: float  # Базовая вероятность

@dataclass
class PatientModel:
    """Модель пациента с уникальными характеристиками"""
    name: str
    description: str
    comfort_mult: float     # Множитель к попаданию в стратегию по комфорту (>1<2)
    # Предпочтительные стратегии (сильные стороны этого пациента)
    preferred_modes: List[BaseMode]  # Номера режимов, которые работают лучше
    history: str #история болезни
    events: List[str]
    accuracy: int 
    stability: int
    comfort: int

class DoshaType(Enum):
    """Типы аюрведических дош"""
    VATA = "Вата"   #ВОЗДУХ Симптомы: Беспокойство, сухость кожи, холод в конечностях, лечение: Теплые, успокаивающие методы (теплое масло, прогрев)
    PITTA = "Питта" #ОГОНЬ Симптомы: Раздражение, жар, покраснение, лечение: Охлаждающие методы (компресс, вентиляция)
    KAPHA = "Капха" #ВОДА Симптомы:Вялость, отеки, заторможенность, лечение: Стимулирующие методы (интенсивная стимуляция)
    WRONG = "Неверный"

class NegativeEvent:
    """Негативное событие (доша-дисбаланс)"""
    def __init__(self, dosha: DoshaType):
        self.dosha = dosha
        self.duration = random.randint(2, 4)  # Длительность в этапах
        self.severity = random.randint(10, 25)  # Сила штрафа
        self.active = True
        
        # Симптомы для каждой доши
        self.symptoms = {
            DoshaType.VATA: [
                "Пациент беспокоен, совершает резкие движения",
                "Кожа сухая и шелушится",
                "Пациент жалуется на холод в конечностях",
                "Заметны мышечные спазмы и подергивания",
                "Дыхание неровное, прерывистое"
            ],
            DoshaType.PITTA: [
                "Кожа покраснела, появилась сыпь",
                "Пациент раздражен, повысилась температура",
                "Повышенное потоотделение",
                "Жалуется на жжение в точках воздействия",
                "Глаза покраснели, воспаленный вид"
            ],
            DoshaType.KAPHA: [
                "Пациент вялый, заторможенный",
                "Появилась отечность тканей",
                "Кожа холодная и липкая",
                "Дыхание поверхностное, медленное",
                "Слизистые выделения, кашель"
            ]
        }
        
        # Лечебные воздействия для каждой доши
        self.remedies = {
            DoshaType.VATA: [
                "теплое масляное обертывание",
                "глубокий прогрев точек",
                "теплое питье"
            ],
            DoshaType.PITTA: [
                "охлаждающий компресс",
                "прохладная вентиляция",
                "крио обертывание",
            ],
            DoshaType.KAPHA: [
                "интенсивная стимуляция",
                "стимуляция сосудов",
                "тонизирующее воздействие"
            ],
            DoshaType.WRONG: [
                "электрошок",
                "заломить руку за спину",
                "рассказать шутку"
            ]
        }
        
        self.current_symptom = random.choice(self.symptoms[dosha])
        self.correct_remedy_options = self.remedies[dosha]
        self.wrong_attempts = 0
        
    def apply_penalty(self, simulator):
        """Применяет штраф за дисбаланс"""
        if not self.active:
            return
            
        penalty = self.severity
        
        if self.dosha == DoshaType.VATA:
            simulator.accuracy = max(0, simulator.accuracy - penalty)
            simulator.stability = max(0, simulator.stability - int(penalty * 0.7))
        elif self.dosha == DoshaType.PITTA:
            simulator.comfort = max(0, simulator.comfort - penalty)
            simulator.energy = max(0, simulator.energy - int(penalty * 0.7))
        elif self.dosha == DoshaType.KAPHA:
            simulator.stability = max(0, simulator.stability - penalty)
            simulator.energy = max(0, simulator.energy - int(penalty * 0.7))
            
        self.duration -= 1
        if self.duration <= 0:
            self.active = False
            return "событие прошло само"
        return None
    
    def display_info(self):
        """Отображает информацию о событии"""
        dosha_colors = {
            DoshaType.VATA: "🌪️",
            DoshaType.PITTA: "🔥",
            DoshaType.KAPHA: "💧"
        }
        
        sprint(f"\n{dosha_colors[self.dosha]} ДИСБАЛАНС ДОШИ: {self.dosha.value}")
        sprint(f"📋 Симптом: {self.current_symptom}")
        sprint(f"⏳ Длительность: {self.duration} этапов")
        sprint(f"💢 Сила: {self.severity}% за этап")
        
    def attempt_cure(self, remedy_choice):
        """Попытка исцелить дисбаланс"""
        if remedy_choice in self.correct_remedy_options:
            self.active = False
            return True, f"✅ Дисбаланс {self.dosha.value} устранен!"
        else:
            self.wrong_attempts += 1
            self.severity += 5  # Усиливаем штраф за неправильную попытку
            return False, f"❌ Неверное воздействие! Дисбаланс усиливается."
            
            
def load_patients_from_json(filename='patients.json'):
    """Загружает пациентов из JSON-файла"""
    with open(filename, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    patients = {}
    for patient_id, patient_data in data.items():
        # Конвертируем preferred_modes из строк в BaseMode
        converted_modes = []
        for mode_list in patient_data['preferred_modes']:
            if isinstance(mode_list, list):
                converted_modes.append([BaseMode.from_string(mode) for mode in mode_list])
            else:
                converted_modes.append([BaseMode.from_string(mode_list)])
        
        patient_data['preferred_modes'] = converted_modes
        patients[patient_id] = PatientModel(**patient_data)
    
    return patients
    
    
def load_positive_events_from_json(filepath: str = 'positive_events.json') -> List[PositiveEvent]:
    """
    Загружает позитивные события из JSON-файла
    Замена для вашего старого кода
    """
    with open(filepath, 'r', encoding='utf-8') as f:
        events_data = json.load(f)
    
    # Создаем объекты PositiveEvent
    positive_events = []
    for event_data in events_data:
        positive_events.append(PositiveEvent(
            name=event_data['name'],
            description=event_data['description'],
            effects=event_data['effects'],
            min_score=event_data['min_score'],
            probability=event_data['probability']
        ))
    
    return positive_events

class SpiderRobotSimulator:
    def __init__(self, patient_id=None):
        # Определяем модели пациентов
        self.patient_models = load_patients_from_json(str(Path(__file__).parent) + '/data/patients.json')
        # Позитивные события
        self.positive_events = load_positive_events_from_json(str(Path(__file__).parent) + '/data/positive_events.json')
        
        # Выбор пациента
        if patient_id is None:
            self.current_patient_key = random.choice(list(self.patient_models.keys()))
        else:
            self.current_patient_key = patient_id
        self.current_patient = self.patient_models[self.current_patient_key]
        
        # Инициализация состояний
        self.accuracy = self.current_patient.accuracy
        self.stability = self.current_patient.stability
        self.comfort = self.current_patient.comfort
        
        self.stage = 1
        self.total_stages = 8
        self.energy = self.total_stages * 10 + 20
        self.completed = False
        
        # Система негативных событий
        self.active_events: List[NegativeEvent] = []
        self.event_cooldown = 0
        self.patient_events = self.current_patient.events
        
        # Статистика
        self.cured_events = 0
        self.failed_events = 0
        self.positive_event_count = 0
    
    def apply_randomness(self, base_value, variation_percent=20):
        """Добавляет случайность к значению"""
        variation = base_value * (variation_percent / 100)
        return base_value + random.uniform(-variation, variation)
    
    def check_for_new_event(self):
        """Проверяет появление нового негативного события"""
        if self.event_cooldown > 0:
            self.event_cooldown -= 1
            return
            
        # Шанс появления события увеличивается при низких показателях
        chance = 0.3
        if self.comfort < 50:
            chance += 0.2
        if self.accuracy < 50:
            chance += 0.1
        if self.stability < 50:
            chance += 0.1
            
        if random.random() < chance and len(self.active_events) < 2:
            dosha = random.choice([DoshaType.VATA, DoshaType.PITTA, DoshaType.KAPHA])
            new_event = NegativeEvent(dosha)
            self.active_events.append(new_event)
            self.event_cooldown = random.randint(2, 4)
            
            sprint(f"\n⚠️  ПОЯВИЛСЯ НОВЫЙ ДИСБАЛАНС!")
            new_event.display_info()
            time.sleep(SLEEP_TIME)
    
    def apply_event_penalties(self):
        """Применяет штрафы от активных событий"""
        expired_events = []
        
        for event in self.active_events:
            if event.active:
                result = event.apply_penalty(self)
                if result:
                    sprint(f"\nℹ️  {result}")
                    expired_events.append(event)
            else:
                expired_events.append(event)
                
        # Удаляем завершенные события
        for event in expired_events:
            if event in self.active_events:
                self.active_events.remove(event)
                if not event.active:  # Если было вылечено
                    self.cured_events += 1
                else:  # Если прошло само
                    self.failed_events += 1
    
    def handle_events(self):
        """Обрабатывает взаимодействие с событиями"""
        if not self.active_events:
            return
            
        sprint("\n" + "="*50)
        sprint("ЛЕЧЕНИЕ ДИСБАЛАНСОВ")
        sprint("="*50)
        
        for i, event in enumerate(self.active_events[:], start=1):
            if event.active:
                sprint(f"\n{i}. Дисбаланс {event.dosha.value}:")
                event.display_info()
                
                # Предлагаем варианты лечения
                all_remedies = []
                for dosha in DoshaType:
                    all_remedies.extend(event.remedies[dosha])
                
                # Выбираем 3 случайных варианта + правильный
                options = random.sample([r for r in all_remedies if r not in event.correct_remedy_options], 3)
                options.append(random.choice(event.correct_remedy_options))
                random.shuffle(options)
                
                sprint("\nВарианты лечения:")
                for idx, remedy in enumerate(options, start=1):
                    sprint(f"  {idx}. {remedy}")
                
                try:
                    choice = sinput("Выберите лечение (1-4) или 0 чтобы пропустить: ")
                    if choice == "0":
                        continue
                        
                    choice_idx = int(choice) - 1
                    if 0 <= choice_idx < 4:
                        selected_remedy = options[choice_idx]
                        success, message = event.attempt_cure(selected_remedy)
                        sprint(f"\n{message}")
                        
                        if success:
                            # Бонус за правильное лечение
                            bonus = 15
                            if event.dosha == DoshaType.VATA:
                                self.accuracy = min(100, self.accuracy + bonus)
                                self.stability = min(100, self.stability + int(bonus * 0.7))
                            elif event.dosha == DoshaType.PITTA:
                                self.comfort = min(100, self.comfort + bonus)
                                self.energy = min(100, self.energy + int(bonus * 0.7))
                            elif event.dosha == DoshaType.KAPHA:
                                self.stability = min(100, self.stability + bonus)
                                self.energy = min(100, self.energy + int(bonus * 0.7))
                    else:
                        sprint("❌ Неверный выбор!")
                except (ValueError, IndexError):
                    sprint("❌ Ошибка ввода!")
                    
                time.sleep(SLEEP_TIME)
    
    def display_status(self):
        """Отображает текущее состояние системы"""
        #sprint("\n" + "="*70)
        sprint(f"ЭТАП {self.stage} из {self.total_stages}")
        
        # Показываем активные события
        if self.active_events:
            #sprint("-"*70)
            sprint("🌡️  АКТИВНЫЕ ДИСБАЛАНСЫ:")
            for event in self.active_events:
                if event.active:
                    dosha_icons = {
                        DoshaType.VATA: "🌪️",
                        DoshaType.PITTA: "🔥",
                        DoshaType.KAPHA: "💧"
                    }
                    sprint(f"  {dosha_icons[event.dosha]} {event.dosha.value}: {event.current_symptom}")
        
        #sprint("-"*70)
        sprint(f"ТОЧНОСТЬ:     [{'█' * (self.accuracy // 5)}{'░' * (20 - self.accuracy // 5)}] {self.accuracy}%")
        sprint(f"СТАБИЛЬНОСТЬ: [{'█' * (self.stability // 5)}{'░' * (20 - self.stability // 5)}] {self.stability}%")
        sprint(f"КОМФОРТ:      [{'█' * (self.comfort // 5)}{'░' * (20 - self.comfort // 5)}] {self.comfort}%")
        sprint(f"ЭНЕРГИЯ:      [{'█' * (self.energy // 5)}{'░' * (20 - self.energy // 5)}] {self.energy}%")
        
        # Статистика по событиям
        if self.cured_events > 0 or self.failed_events > 0:
            sprint(f"📊 Излечено дисбалансов: {self.cured_events}")
        
        #sprint("="*70)
        
        # Визуальные предупреждения
        warnings = []
        if self.accuracy < 30:
            warnings.append("⚠️  Точность критически низка!")
        if self.stability < 30:
            warnings.append("⚠️  Стабильность под угрозой!")
        if self.comfort < 30:
            warnings.append("⚠️  Пациент испытывает дискомфорт!")
        if self.energy < 30:
            warnings.append("⚠️  Энергия на исходе!")
        
        if warnings:
            sprint("\n" + "\n".join(warnings))
    
    def update_score(self):
        """Обновляет текущий счет на основе состояния параметров"""
        score = self.comfort * 2 + self.accuracy * 0.5 + self.stability * 0.5 + self.energy * 0.3
        self.current_score = int(score)
    
    def check_positive_event(self):
        """Проверяет и применяет позитивное событие"""
        self.update_score()
        
        # Фильтруем доступные события по минимальному счету
        available_events = [event for event in self.positive_events 
                          if self.current_score >= event.min_score]
        
        if not available_events:
            return False
        
        # Рассчитываем общую вероятность (увеличивается с ростом счета)
        base_probability = min(0.3, 0.05 + (self.current_score / 1000))
        
        # Увеличиваем вероятность после каждого события
        if self.positive_event_count > 0:
            base_probability *= (1 + self.positive_event_count * 0.1)
        
        if random.random() > base_probability:
            return False
        
        # Выбираем событие с учетом весов (более редкие события имеют меньший вес)
        weights = [1 / (event.probability * 10) for event in available_events]
        chosen_event = random.choices(available_events, weights=weights, k=1)[0]
        
        # Применяем эффекты события
        sprint(f"\n✨ ПОЗИТИВНОЕ СОБЫТИЕ: {chosen_event.name} ✨")
        sprint(f"📖 {chosen_event.description}")
        
        for param, change in chosen_event.effects.items():
            if param == "accuracy":
                self.accuracy = min(100, self.accuracy + change)
                sprint(f"  📊 Точность: +{change}%")
            elif param == "stability":
                self.stability = min(100, self.stability + change)
                sprint(f"  ⚖️ Стабильность: +{change}%")
            elif param == "comfort":
                self.comfort = min(100, self.comfort + change)
                sprint(f"  😌 Комфорт: +{change}%")
            elif param == "energy":
                self.energy = min(100, self.energy + change)
                sprint(f"  ⚡ Энергия: +{change}%")
        
        self.positive_event_count += 1
        time.sleep(SLEEP_TIME)
        return True
    
    def apply_mode(self, mode):
        """Применяет выбранный режим работы"""
        sprint("\n" + "—" * 50)
        
        mode_names = {BaseMode.FAST: BaseModeCaptions[0], BaseMode.NORMAL: BaseModeCaptions[1], BaseMode.SLOW: BaseModeCaptions[2]}
        mode_icons = {BaseMode.FAST: "🔴", BaseMode.NORMAL: "🟡", BaseMode.SLOW: "🟢"}
        
        sprint(f"{mode_icons[mode]} РЕЖИМ: {mode_names[mode]}")
        
        # Базовые эффекты
        base_effects = {
            BaseMode.FAST:   {"accuracy": -10, "stability": -5, "comfort": 0, "energy": -14},
            BaseMode.NORMAL: {"accuracy":  -7, "stability": -7, "comfort": 0, "energy": -12},
            BaseMode.SLOW:   {"accuracy":  -5, "stability": -10, "comfort": 0, "energy": -10}
        }
        
        # Рассчитываем фактические эффекты
        actual_effects = {}
        for param, base_effect in base_effects[mode].items():
            randomized_effect = self.apply_randomness(base_effect, 15)
            actual_effects[param] = int(round(randomized_effect))
        
        # Определение стратегии
        if len(self.current_patient.preferred_modes) > 0:
            preff_modes = self.current_patient.preferred_modes.pop(0)
            if mode in preff_modes:
                comfort_bonus = int(self.current_patient.comfort_mult * 10)
            else:
                comfort_bonus = int(-self.current_patient.comfort_mult * 5)
        else:
            comfort_bonus = 0
        
        actual_effects["comfort"] = comfort_bonus
        
        # Описания действий
        actions = {
            BaseMode.FAST: "Робот быстро находит точку и вводит иглу...",
            BaseMode.NORMAL: "Робот тщательно сканирует и калибрует давление...",
            BaseMode.SLOW: "Игла мягко вибрирует и прогревает точку..."
        }
        
        sprint(actions[mode])
        
        # Применяем эффекты
        self.accuracy = max(0, min(100, self.accuracy + actual_effects["accuracy"]))
        self.stability = max(0, min(100, self.stability + actual_effects["stability"]))
        self.comfort = max(0, min(100, self.comfort + actual_effects["comfort"]))
        self.energy = max(0, min(100, self.energy + actual_effects["energy"]))
        
        # Случайное событие от пациента
        if random.random() > 0.4:
            event = random.choice(self.patient_events)
            sprint(f"\n💭 {event}")
        
        # Применяем штрафы от событий
        self.apply_event_penalties()
        
        time.sleep(SLEEP_TIME)
    
    def run_session(self):
        """Запускает игровую сессию"""
        #sprint("="*70)
        sprint("НЕЙРОМАСТЕР 3000: ПОДКЛЮЧЕНИЕ")
        #sprint("="*70)
        sprint(f"\n💊 ПАЦИЕНТ: {self.current_patient.name}")
        sprint(f"📋 {self.current_patient.description}")
        sprint(f"\nИстория болезни:\n{self.current_patient.history}")
        
        
        # Информация о системе дош
        #sprint("\n" + "="*70)
        #sprint("СИСТЕМА ДИСБАЛАНСОВ (Тридоша):")
        #sprint("-"*70)
        #sprint("🌪️  ВАТА (Воздух): беспокойство, сухость, холод")
        #sprint("🔥  ПИТТА (Огонь): раздражение, жар, воспаление")
        #sprint("💧  КАПХА (Вода): вялость, отеки, застой")
        #sprint("\nПри появлении дисбаланса нужно выбрать правильное лечение!")
        #sprint("="*70)
        
        sprint("\nНажмите Enter чтобы начать...")
        sinput()
        
        while self.stage <= self.total_stages and not self.completed:
            sclear()
            
            # Проверяем новое событие в начале этапа
            self.check_for_new_event()
            
            self.display_status()
            
            # Обработка активных событий
            if self.active_events:
                self.handle_events()
                time.sleep(SLEEP_TIME)
            
            # Выбор основного действия
            sprint("\nВыберите режим для этапа:")
            sprint(f"1. 🔴 {BaseModeCaptions[0]}")
            sprint(f"2. 🟡 {BaseModeCaptions[1]}")
            sprint(f"3. 🟢 {BaseModeCaptions[2]}")
            
            try:
                smode = sinput("Ваш выбор (1-3): ")
                if smode == "forcewin":
                    #sprint("\n" + "="*70)
                    sprint("СЕАНС УСПЕШНО ЗАВЕРШЁН!")
                    #sprint("="*70)
                    grade, score = self.calculate_score()
                    sprint(f"\n🎯 ИТОГОВАЯ ОЦЕНКА: {grade}")
                    sprint(f"📊 НАБРАНО ОЧКОВ: {score:.1f}")
                    sprint(f"✨ ПОЗИТИВНЫХ СОБЫТИЙ: {self.positive_event_count}")
                    return True
                elif smode == "forceloose":
                    grade, score = self.calculate_score()
                    sprint(f"\nИгра окончена. Результат: {grade} (Очки: {score:.1f})")
                    sprint(f"✨ ПОЗИТИВНЫХ СОБЫТИЙ: {self.positive_event_count}")
                    sinput()
                    return False
                elif smode == "forceall":
                    self.comfort  = 100
                    self.accuracy = 100
                    self.stability = 100
                    self.energy = 100
                    continue
                    
                mode = int(smode)
                if mode not in [BaseMode.FAST.value, BaseMode.NORMAL.value, BaseMode.SLOW.value]:
                    sprint("Ошибка: выберите 1, 2 или 3")
                    sinput()
                    continue
            except ValueError:
                sprint("Ошибка: введите число")
                sinput()
                continue
            
            self.apply_mode(BaseMode(mode))
            self.stage += 1
            
            # Проверка на завершение
            if self.check_game_over_bad():
                grade, score = self.calculate_score()
                sprint(f"\nИгра окончена. Результат: {grade} (Очки: {score:.1f})")
                sprint(f"✨ ПОЗИТИВНЫХ СОБЫТИЙ: {self.positive_event_count}")
                return False
            
            # Специальное действие
            if self.stage <= self.total_stages:
                sprint("\nСпециальное действие перед следующим этапом:")
                sprint(f"1. ⚙️ {SpecModeCaptions[SpecMode.CALIBRATE.value-1]}")
                sprint(f"2. 📐 {SpecModeCaptions[SpecMode.STABILIZE.value-1]}")
                sprint(f"3. ⏸️  {SpecModeCaptions[SpecMode.PAUSE_POW.value-1]}")
                sprint("4. ⏭️  Пропустить")
                
                try:
                    action = int(sinput("Ваш выбор (1-4): "))
                    if action == 4:
                        continue
                    elif action in [SpecMode.CALIBRATE.value, SpecMode.STABILIZE.value, SpecMode.PAUSE_POW.value]:
                        self.apply_special_action(action)
                        
                        # Применяем штрафы после специального действия
                        self.apply_event_penalties()
                        
                        if self.check_game_over_bad():
                            grade, score = self.calculate_score()
                            sprint(f"\nИгра окончена. Результат: {grade} (Очки: {score:.1f})")
                            sprint(f"✨ ПОЗИТИВНЫХ СОБЫТИЙ: {self.positive_event_count}")
                            sinput()
                            return False
                    else:
                        sprint("Ошибка: выберите от 1 до 4")
                        sinput()
                except ValueError:
                    sprint("Ошибка: введите число")
                    sinput()
        
        # Завершение игры
        if not self.check_game_over_bad():
            #sprint("\n" + "="*70)
            sprint("СЕАНС УСПЕШНО ЗАВЕРШЁН!")
            #sprint("="*70)
            
            grade, score = self.calculate_score()
            
            # Бонусы за лечение дисбалансов
            event_bonus = self.cured_events * 25
            score_with_bonus = score + event_bonus
            
            sprint(f"\n🎯 ИТОГОВАЯ ОЦЕНКА: {grade}")
            sprint(f"📊 Базовых очков: {score:.1f}")
            sprint(f"✨ ПОЗИТИВНЫХ СОБЫТИЙ: {self.positive_event_count}")
            if self.cured_events > 0:
                sprint(f"🏥 Бонус за лечение дисбалансов: +{event_bonus} (x{self.cured_events})")
            sprint(f"💰 ИТОГО ОЧКОВ: {score_with_bonus:.1f}")
            
            return True
            
    def check_game_over_bad(self):
        """Проверяет условия завершения игры"""
        failure_reasons = []
        
        if self.accuracy <= 0:
            failure_reasons.append("Робот совершил критическую ошибку позиционирования!")
        if self.stability <= 0:
            failure_reasons.append("Робот потерял устойчивость и упал!")
        if self.comfort <= 0:
            failure_reasons.append("Пациенту стало плохо, процедура прервана!")
        if self.energy <= 0:
            failure_reasons.append("Робот отключился от нехватки энергии!")
        
        if failure_reasons:
            sprint("\n💥 КАТАСТРОФА!")
            for reason in failure_reasons:
                sprint(f"  • {reason}")
            return True
        return False
    
    def calculate_score(self):
        """Рассчитывает итоговую оценку с учетом пациента"""
        base_score = 0
        
        # Базовые баллы за комфорт
        base_score += self.comfort * 2
        
        # Бонусы за хорошее состояние других параметров
        base_score += self.accuracy * 0.5
        base_score += self.stability * 0.5
        base_score += self.energy * 0.3
        
        # Бонус за позитивные события
        base_score += self.positive_event_count * 15
        
        # Множитель за соответствие типу пациента
        patient_multiplier = 1.0
        if self.comfort > 80:
            patient_multiplier = 1.2  # Бонус за хорошую работу с пациентом
        
        final_score = base_score * patient_multiplier
        
        # Определяем буквенную оценку
        if final_score > 350 and self.comfort > 90:
            return "S (Идеально!) 🌟", final_score
        elif final_score > 280 and self.comfort > 80:
            return "A (Отлично) 👍", final_score
        elif final_score > 220 and self.comfort > 70:
            return "B (Хорошо) ✅", final_score
        elif final_score > 180:
            return "C (Удовлетворительно) ⚠️", final_score
        else:
            return "F (Провал) ❌", final_score
    
    def apply_special_action(self, action):
        """Применяет специальное действие между этапами"""
        sprint("\n" + "—" * 50)
        
        action_names = {SpecMode.CALIBRATE.value: SpecModeCaptions[SpecMode.CALIBRATE.value-1], 
                        SpecMode.STABILIZE.value: SpecModeCaptions[SpecMode.STABILIZE.value-1], 
                        SpecMode.PAUSE_POW.value: SpecModeCaptions[SpecMode.PAUSE_POW.value-1]}
        action_icons = {SpecMode.CALIBRATE.value: "⚙️ ", 
                        SpecMode.STABILIZE.value: "📐", 
                        SpecMode.PAUSE_POW.value: "⏸️ "}
        
        sprint(f"{action_icons[action]} {action_names[action]}")
        
        # Базовые эффекты
        base_effects = {
            SpecMode.CALIBRATE.value: {"accuracy": 20, "stability": -5, "comfort": -5,  "energy": -5},
            SpecMode.STABILIZE.value: {"stability": 20, "accuracy": -5, "comfort": -10, "energy": -10},
            SpecMode.PAUSE_POW.value: {"stability": -5, "accuracy": -5, "comfort": 10,  "energy": 15}
        }
        
        # Рассчитываем с учетом пациента (для спецдействий множитель комфорта важен)
        actual_effects = {}
        for param, base_effect in base_effects[action].items():
            randomized_effect = self.apply_randomness(base_effect, 15)
            # Округляем для удобства
            actual_effects[param] = int(round(randomized_effect))
        
        # Описания
        descriptions = {
            SpecMode.CALIBRATE.value: "Робот проводит самодиагностику и настройку...",
            SpecMode.STABILIZE.value: "Ноги робота перераспределяют нагрузку...",
            SpecMode.PAUSE_POW.value: "Даем пациенту время адаптироваться..."
        }
        
        sprint(descriptions[action])
        
        # Применяем эффекты
        for param, change in actual_effects.items():
            if param == "accuracy":
                self.accuracy = max(0, min(100, self.accuracy + change))
            elif param == "stability":
                self.stability = max(0, min(100, self.stability + change))
            elif param == "comfort":
                self.comfort = max(0, min(100, self.comfort + change))
            elif param == "energy":
                self.energy = max(0, min(100, self.energy + change))
        
        # Проверяем позитивное событие после специального действия
        self.check_positive_event()
        
        time.sleep(SLEEP_TIME)

# Запуск симулятора
if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Нейромастер. Часть 2')
    print(f"Нейромастер. Часть 2. Геймплей")
    
    
    # Optional argument with a default value
    parser.add_argument('--plain', action='store_true', help='Использовать простую консоль')
    parser.add_argument('--debug', action='store_true', help='Выводить отладочный текст')
    parser.add_argument('--log', action='store_true', help='Сохранять результат в `log.txt`')


    # 3. Parse the arguments
    args = parser.parse_args()
    
    init_sumulator(args.plain,args.debug,args.log,False,str(Path(__file__).parent)+"/user")
    
    #sprint("="*70)
    sprint("МЕДИЦИНСКИЙ СИМУЛЯТОР: ОПЕРАТОР РОБОТА-ПАУКА")
    #sprint("="*70)
    
    while True:
        simulator = SpiderRobotSimulator()
        simulator.run_session()
        
        replay = sinput("🎮 Хотите попробовать с другим пациентом? (да/нет): ").lower()
        if replay not in ['да', 'д', 'yes', 'y', '1']:
            sprint("\nСпасибо за игру! Удачных операций!")
            break
