import wx
import re
import threading
import queue
import time
import os
import sys
from rich.console import Console
from rich.markdown import Markdown
import logging

PLAIN_MODE = False
DEBUG_TEXT = False
USE_LOG_FILE = False
USE_WX_WIDGET = False  # Новая опция для использования wx виджетов
MANUAL_TEXT_MD = "<NONE>"

console = None
wx_app = None
wx_frame = None
input_queue = queue.Queue()
output_queue = queue.Queue()

def keep_only_alphanumeric_utf8(text):
    cleaned_text = re.sub(r'[^\w\s\-\\\'\/\.\,\!\?]', '', text, flags=re.UNICODE)
    return cleaned_text

class SimpleIOPanel(wx.Panel):
    def __init__(self, parent):
        super().__init__(parent)
       
        # Вертикальный sizer для компоновки
        sizer = wx.BoxSizer(wx.VERTICAL)
       
        # Многострочное текстовое поле для вывода (только чтение)
        self.output_text = wx.TextCtrl(self, style=wx.TE_MULTILINE | wx.TE_READONLY | wx.TE_RICH2)
        self.output_text.SetBackgroundColour(wx.Colour(240, 240, 240))
        font = wx.Font(10, wx.FONTFAMILY_TELETYPE, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL)
        self.output_text.SetFont(font)
       
        sizer.Add(self.output_text, 1, wx.EXPAND | wx.ALL, 5)
       
        # Горизонтальный sizer для поля ввода и кнопки
        input_sizer = wx.BoxSizer(wx.HORIZONTAL)
       
        # Метка для поля ввода
        input_label = wx.StaticText(self, label="Ввод:")
        input_sizer.Add(input_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 5)
       
        # Однострочное поле ввода
        self.input_text = wx.TextCtrl(self, style=wx.TE_PROCESS_ENTER)
        self.input_text.Bind(wx.EVT_TEXT_ENTER, self.on_enter)
        input_sizer.Add(self.input_text, 1, wx.EXPAND | wx.RIGHT, 5)
       
        # Кнопка отправки
        self.send_button = wx.Button(self, label="Отправить")
        self.send_button.Bind(wx.EVT_BUTTON, self.on_enter)
        input_sizer.Add(self.send_button, 0)
       
        sizer.Add(input_sizer, 0, wx.EXPAND | wx.ALL, 5)
       
        self.SetSizer(sizer)
       
        # Таймер для периодической проверки очереди вывода
        self.timer = wx.Timer(self)
        self.Bind(wx.EVT_TIMER, self.on_timer, self.timer)
        self.timer.Start(100)  # Проверять каждые 100 мс
       
    def on_enter(self, event):
        """Обработка ввода (Enter или кнопка)"""
        text = self.input_text.GetValue().strip()
        if text:
            # Помещаем в очередь ввода
            input_queue.put(text)
            self.input_text.Clear()
        else:
            input_queue.put("\n")
       
    def on_timer(self, event):
        """Проверка очереди вывода и обновление текстового поля"""
        try:
            while True:
                text = output_queue.get_nowait()
                self.output_text.AppendText(text + "\n")
               
                # Автопрокрутка вниз
                self.output_text.ShowPosition(self.output_text.GetLastPosition())
        except queue.Empty:
            pass

class SimpleIOFrame(wx.Frame):
    def __init__(self):
        super().__init__(None, title="Ввод/Вывод", size=(600, 400))
       
        self.panel = SimpleIOPanel(self)
       
        # Создаем меню
        menubar = wx.MenuBar()
       
        # Меню Файл
        file_menu = wx.Menu()
        clear_item = file_menu.Append(wx.ID_CLEAR, "Очистить вывод\tCtrl+L")
        exit_item = file_menu.Append(wx.ID_EXIT, "Выход\tCtrl+Q")
       
        menubar.Append(file_menu, "&Файл")
        self.SetMenuBar(menubar)
       
        # Привязка событий меню
        self.Bind(wx.EVT_MENU, self.on_clear, clear_item)
        self.Bind(wx.EVT_MENU, self.on_exit, exit_item)
       
        # Горячие клавиши
        accel_tbl = wx.AcceleratorTable([
            (wx.ACCEL_CTRL, ord('L'), wx.ID_CLEAR),
            (wx.ACCEL_CTRL, ord('Q'), wx.ID_EXIT)
        ])
        self.SetAcceleratorTable(accel_tbl)
       
        self.Show()
       
    def on_clear(self, event):
        """Очистка поля вывода"""
        self.panel.output_text.Clear()
       
    def on_exit(self, event):
        """Выход из программы"""
        print("Окно закрывается")
        self.Close()
        # Явно выходим из приложения
        wx.GetApp().ExitMainLoop()
        sys.exit(0)
        #wx.Exit()  # или self.Destroy() + 
        # или event.Skip() если нужно стандартное поведение

def wx_thread_func():
    """Функция для запуска wx приложения в отдельном потоке"""
    global wx_app, wx_frame
   
    wx_app = wx.App(False)
    wx_frame = SimpleIOFrame()
    wx_app.MainLoop()

def sprint(s):
    """Умная печать на консоль с учетом настроек вывода"""
    global PLAIN_MODE, USE_LOG_FILE, console, USE_WX_WIDGET
   
    if USE_WX_WIDGET:
        # Отправляем текст в очередь для wx виджета
        output_queue.put(str(s))
    elif PLAIN_MODE:
        print(keep_only_alphanumeric_utf8(s))
    else:
        console.print(s)
       
    if USE_LOG_FILE:
        logging.info(keep_only_alphanumeric_utf8(s))

def sinput(s=""):
    """Умный ввод с консоли или wx виджета"""
    global USE_LOG_FILE, PLAIN_MODE, console, USE_WX_WIDGET
   
    if USE_WX_WIDGET:
        # Если есть текст приглашения, выводим его
        if s:
            sprint(s)
       
        # Ждем ввод из wx виджета
        while True:
            try:
                res = input_queue.get(block=True, timeout=0.1)
               
                # Обработка специальных команд
                if res.strip().startswith('forcewin'):
                    return "forcewin"
                elif res.strip().startswith('forceloose'):
                    return "forceloose"
                elif res.strip().startswith('forceall'):
                    return "forceall"
                elif res.strip().startswith('help'):
                    manual_text = "Справка по командам..."
                    md = Markdown(manual_text)
                    return md
                   
                # Если строка начинается с *, записываем в лог
                if res.strip().startswith('*'):
                    if USE_LOG_FILE:
                        logging.info(f"COMMENT: {res}")
                    continue  # Повторный запрос ввода
                else:
                    if USE_LOG_FILE:
                        logging.info(keep_only_alphanumeric_utf8(s) + "\n>" + res)
                    return res
            except queue.Empty:
                # Даем время для обработки событий wx
                time.sleep(0.01)
                continue
   
    if PLAIN_MODE:
        print(keep_only_alphanumeric_utf8(s) + " >")
    # Оригинальный код для консольного ввода
    while True:
        if PLAIN_MODE:
            res = input()
        else:
            res = console.input(s)
           
        if res.strip().startswith('forcewin'):
            return "forcewin"
        elif res.strip().startswith('forceloose'):
            return "forceloose"
        elif res.strip().startswith('forceall'):
            return "forceall"
        elif res.strip().startswith('help'):
            manual_text = "Справка по командам..."
            if PLAIN_MODE:
                print(keep_only_alphanumeric_utf8(MANUAL_TEXT_MD))
            else:
                md = Markdown(MANUAL_TEXT_MD)
                console.print(md)
            input("\nВВОД для повторного ввода...")
            continue
           
        # Если строка начинается с *, записываем в лог и запрашиваем ввод снова
        if res.strip().startswith('*'):
            if USE_LOG_FILE:
                logging.info(f"COMMENT: {res}")
            continue  # Повторный запрос ввода
        else:
            if USE_LOG_FILE:
                logging.info(keep_only_alphanumeric_utf8(s) + "\n>" + res)
            break
    return res

def sclear():
    """Умная очистка консоли или wx виджета"""
    global PLAIN_MODE, console, USE_WX_WIDGET
   
    if USE_WX_WIDGET:
        global wx_app, wx_frame
        # Очищаем очередь вывода
        with output_queue.mutex:
            output_queue.queue.clear()
        wx_frame.panel.output_text.Clear()
        # Отправляем команду очистки (может потребоваться дополнительная логика)
        pass
    elif not PLAIN_MODE:
        console.clear()
    else:
        if os.name == 'nt':
            command = 'cls' 
        else:
            command = 'clear'
        os.system(command)

def init_sumulator(use_plain_mode, use_debug_text, use_log_file, use_wx_widget, log_path):
    global console, PLAIN_MODE, DEBUG_TEXT, USE_LOG_FILE, USE_WX_WIDGET, MANUAL_TEXT_MD
   
    PLAIN_MODE = use_plain_mode
    DEBUG_TEXT = use_debug_text
    USE_LOG_FILE = use_log_file
    USE_WX_WIDGET = use_wx_widget
    with open(log_path+"/game_manual.md", 'r', encoding='utf-8') as file:
        MANUAL_TEXT_MD = file.read()
   
    if USE_WX_WIDGET:
        # Запускаем wx приложение в отдельном потоке
        wx_thread = threading.Thread(target=wx_thread_func, daemon=True)
        wx_thread.start()
       
        # Даем время на инициализацию GUI
        time.sleep(0.5)
    elif not PLAIN_MODE:
        console = Console(width=80)
       
    if USE_LOG_FILE:
        logging.basicConfig(
            filename=log_path+'log.txt',
            level=logging.INFO,
            format='%(message)s',
            filemode='a'
        )

        
def finish_gameio():
    """Окончание игры, аккуратное"""
    global wx_frame, USE_WX_WIDGET
    if USE_WX_WIDGET:
         # Безопасное закрытие из любого потока
        wx.CallAfter(wx_frame.Close)
        time.sleep(2)

# Пример использования:
if __name__ == "__main__":
    # Инициализация с wx виджетом
    init_sumulator(
        use_plain_mode=False,
        use_debug_text=False,
        use_log_file=False,
        use_wx_widget=True  # Включаем wx виджет
    )
   
    # Пример работы
    sprint("Добро пожаловать в программу!")
    sprint("=" * 50)
   
    name = sinput("Введите ваше имя: ")
    sprint(f"Привет, {name}!")
   
    age = sinput("Введите ваш возраст: ")
    sprint(f"Вам {age} лет.")
   
    # Закрытие (в реальной программе по необходимости)
    # close_wx_widget()